<?php

namespace App\Http\Controllers\Api\Dentist;

use App\Models\RFQ;
use App\Models\Cart;
use App\Models\Spec;
use App\Models\User;
use App\Models\Admin;
use App\Models\Order;
use App\Models\Product;
use App\Models\RfqItem;
use App\Models\OrderItem;
use App\Models\RfqsSpecs;
use App\helpers\Attachment;
use App\Services\FcmService;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use App\Http\Controllers\Controller;
use App\Http\Resources\OrderResource;
use Symfony\Component\HttpFoundation\Response;

class OrderController extends Controller
{
    use HasCrudActions;

    protected $model = RFQ::class;

    protected $resource = OrderResource::class;



    public function index(Request $request)
    {
        // Initialize the search criteria based on the request
        $searchableFields = [
            'status' => ['status' => $request->status],
            'order_type' => ['order_type' => $request->type]
        ];
    
        // Initialize the query builder
        $modelResource = $this->search()->where('user_id' , auth()->user()->id)->searchRelated($searchableFields)
            ->when($request->has('date'), function ($q) use ($request) {
                $q->whereDate('created_at', $request->date);
            })
            ->with(['rfqItems.product', 'user:id,owner_name', 'branch:id,name,location', 'product:id,name'])
            ->when(!$request->has('status'), function ($q) {
                $q->where('status', '!=', 'completed');
            });
    
        // Return the response
        return $this->apiResponse('200', __('success'), '', OrderResource::collection($modelResource->orderBy('id', 'DESC')->get()->makeHidden(['image'])));
    }
    
    public function checkout(Request $request, FcmService $fcmService)
    {
        $request->validate([
            'branch_id' => ['required_if:order_type,direct_order', 'exists:branches,id'],
            'order_type' => ['required', 'in:rfq_order,direct_order'],
            'rfq_id' => 'required_if:order_type,rfq_order',
            'payment_method' => ['required', 'in:Cash On Delivery,Bank Account,card'],
            'specs' => ['array'],
            'specs.*' => ['exists:specs,id'],
            'card_id' => ['required_if:payment_method,card'],
            'image' => ['required_if:payment_method,Bank Account'],
        ]);
    
        if ($request->order_type === 'rfq_order') {
            $rfq = RFQ::find($request->rfq_id);
            $rfq->update([
                'payment_method' => $request->payment_method,
                'bank_account_id' => $request->bank_account_id,
                'card_id' => $request->card_id,
            ]);

                        return $this->apiResponse('200', __('success'),  'Order Placed Successfully');

        } else {
            // Get the authenticated user's cart
            $cart = Cart::with('items.product')->where('user_id', auth()->id())->first();
    
            // Check if the cart is empty
            if (!$cart || $cart->items->isEmpty()) {
                return response()->json(['message' => 'Cart is empty'], 400);
            }
    
            // Calculate the total price and total quantity
            $totalPrice = $cart->items->sum(function ($item) {
                return $item->quantity * $item->product->price;
            });
    
            $totalQuantity = $cart->items->sum('quantity');
    
            // Create the RFQ (order)
            $rfq = RFQ::create([
                'branch_id' => $request->branch_id,
                'user_id' => auth()->id(),
                'order_type' => 'direct_order',
                'price' => $totalPrice,
                'qty' => $totalQuantity,
                'status' => 'pending', // Initial status
                'payment_method' => $request->payment_method,
                'card_id' => $request->card_id,
            ]);
    
            if ($request->hasFile('image')) {
                Attachment::addAttachment($request->file('image'), $rfq, 'rfqs/images', ['save' => 'original', 'usage' => 'image']);
            }
    
            // Create RFQ items for each cart item
            foreach ($cart->items as $item) {
                $specs = $request->specs ?? []; // Default to an empty array if no specs are provided for this product
                if ($request->has('specs')) {
                    $specsValues = Spec::find($specs);
                    $specsData = $specsValues->map(function ($specValue) {
                        return [
                            'id' => $specValue->id,
                            'key' => $specValue->key,
                            'value' => $specValue->value,
                        ];
                    })->toArray();
                } else {
                    // If no specs provided, set an empty array for specsData
                    $specsData = [];
                }
    
                RfqItem::create([
                    'rfq_id' => $rfq->id,
                    'product_id' => $item->product_id,
                    'specs' => json_encode($specsData),
                    'qty' => $item->quantity,
                ]);
            }
    
            if ($request->has('specs')) {
                $rfq->specs()->attach($request->specs);
            }
    
            // Clear the user's cart
            $cart->items()->delete();
            $cart->delete();
    
    
        $admins = Admin::with('fcmTokens')->get();
        $successCount = 0;
        $failureCount = 0;
    
        // Loop through all admins
        foreach ($admins as $admin) {
            foreach ($admin->fcmTokens as $token) {
                if ($token->device_token) {
                    $fcmResponse = $fcmService->send(
                        $token->device_token,
                        __('New Order Created'),
                        __('A new direct order has been placed. Order ID: :order_id', ['order_id' => $rfq->id]),
                        [
                            'order_id' => (string) $rfq->id,
                            'action' => 'new_order',
                        ],
                        $admin
                    );
    
                    // Track success and failure counts
                    if ($fcmResponse->successful()) {
                        $successCount++;
                    } else {
                        $failureCount++;
                    }
                }
            }
        }
    
        // Return appropriate response
        if ($successCount > 0) {
            return $this->apiResponse('200', __('success'),  'Order Placed Successfully');

        } else {
            return $this->apiResponse(
                Response::HTTP_INTERNAL_SERVER_ERROR,
                __('error'),
                __('Failed to send notifications to admins')
            );
        }
    
     
    
        
    }
}
    

    public function show($id)
    {
        $order = RFQ::with(['rfqItems.product', 'user:id,owner_name', 'branch:id,name'])->find($id);

        if (!$order) {
            return response()->json(['message' => 'Order not found'], 404);
        }

        return $this->apiResponse('200', __('success'), '', new OrderResource($order));
    }
}
