<?php

namespace App\Http\Controllers\Api\Admin;

use Carbon\Carbon;
use App\Models\RFQ;
use App\Models\User;
use App\Models\Product;
use App\Models\Setting;
use App\Models\BankAccount;
use App\Traits\HasCrudActions;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\Api\BankAccountRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Requests\Api\Admin\SettingRequest;
use App\Http\Resources\Api\Admin\SettingResource;
use App\Http\Resources\Api\vendor\ProductResource;
use App\Http\Resources\Admin\TopSoldProductResource;

class SettingController extends Controller
{
    use HasCrudActions;

    protected $model = Setting::class;

    protected $resource = SettingResource::class;

    protected $request = SettingRequest::class;

    protected $hasPagination = false;


    public function index()
    {
        $record = Setting::first();

        return $this->apiResponse(Response::HTTP_OK, __('success'), __('loaded successfully'), SettingResource::make($record));
    }


    public function update(SettingRequest $request)
    {

        $record = Setting::first();

        $record->update($request->all());

        return $this->apiResponse(Response::HTTP_CREATED, __('success'), __('created successfully'), SettingResource::make($record));
    }

    public function statistics()
    {
        // Count of dentists and vendors
        $dentists = User::where('type', 'dentist')->count();
        $vendors = User::where('type', 'vendor')->count();

        // Daily counts for RFQ and direct orders
        $dailyRfqCount = RFQ::where('order_type', 'rfq_order')
            ->whereDate('created_at', Carbon::today())
            ->count();

        $dailyOrdersCount = RFQ::where('order_type', 'direct_order')
            ->whereDate('created_at', Carbon::today())
            ->count();

        // Total counts for RFQ and direct orders
        $totalRfqCount = RFQ::where('order_type', 'rfq_order')->where('status' , 'complete')->count();
        $totalOrdersCount = RFQ::where('order_type', 'direct_order')->where('status' , 'complete')->count();


        $totalRfqRevenue = RFQ::where('order_type', 'rfq_order')->where('status' , 'complete')->sum('price');
        $totalOrdersRevenue = RFQ::where('order_type', 'direct_order')->where('status' , 'complete')->sum('price');

        // Monthly data grouped by year-month
        $monthlyData = RFQ::select(
            DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
            DB::raw('SUM(CASE WHEN order_type = "rfq_order" THEN 1 ELSE 0 END) as rfq_order_count'),
            DB::raw('SUM(CASE WHEN order_type = "direct_order" THEN 1 ELSE 0 END) as direct_order_count')
        )
            ->groupBy('month')
            ->orderBy('month', 'desc')
            ->get();


        $monthlyRevenueData = RFQ::select(
            DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
            DB::raw('SUM(CASE WHEN order_type = "rfq_order" THEN 1 ELSE 0 END) as rfq_order_count'),
            DB::raw('SUM(CASE WHEN order_type = "direct_order" THEN 1 ELSE 0 END) as direct_order_count')
        )
            ->groupBy('month')
            ->orderBy('month', 'desc')
            ->get();


        $monthlyRevenueData = RFQ::select(
            DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
            DB::raw('SUM(CASE WHEN order_type = "rfq_order" THEN price ELSE 0 END) as rfq_order_revenue'),
            DB::raw('SUM(CASE WHEN order_type = "direct_order" THEN price ELSE 0 END) as direct_order_revenue')
        )
            ->groupBy('month')
            ->orderBy('month', 'desc')
            ->get();

        $topProducts = Product::with('mainCategory')
            ->select('products.id', 'products.name', 'products.price', 'products.description', 'products.main_category_id')
            ->leftJoin('rfq_items', 'products.id', '=', 'rfq_items.product_id')
            ->leftJoin('r_f_q_s', 'products.id', '=', 'r_f_q_s.product_id')
            ->selectRaw('COUNT(rfq_items.product_id) + COUNT(r_f_q_s.product_id) as rfq_count')
            ->groupBy('products.id', 'products.name', 'products.price', 'products.description', 'products.main_category_id')
            ->having('rfq_count', '>', 0) // Only get products with a count greater than 0
            ->orderByDesc('rfq_count')
            ->limit(5)
            ->get();

        $topVendor = RFQ::where('status', 'completed') // Filter by completed status
            ->select('vendor_id', DB::raw('COUNT(*) as completed_rfq_count'))
            ->groupBy('vendor_id')
            ->orderByDesc('completed_rfq_count')
            ->with('vendor') // Assuming a relationship exists between RFQ and User for the vendor
            ->first();


        $topdentist = RFQ::where('status', 'completed') // Filter by completed status
            ->select('user_id', DB::raw('COUNT(*) as completed_rfq_count'))
            ->groupBy('user_id')
            ->orderByDesc('completed_rfq_count')
            ->with('user') // Assuming a relationship exists between RFQ and User for the vendor
            ->first();


        return $this->apiResponse(Response::HTTP_CREATED, __('success'), __('Success'), [
            'dentists_count' => $dentists,
            'vendors_count' => $vendors,
            'daily_rfq_count' => $dailyRfqCount,
            'daily_orders_count' => $dailyOrdersCount,

            'total_rfq_count' => $totalRfqCount,
            'total_orders_count' => $totalOrdersCount,
            'total_rfq_revenue' => $totalRfqRevenue,
            'total_orders_revenue' => $totalOrdersRevenue,

            'monthly_data' => $monthlyData,
            'monthlyRevenueData' => $monthlyRevenueData,
            'topProducts' => TopSoldProductResource::collection($topProducts), // Wrap in resource collection
            'topVendorSeller' => $topVendor ? [
                'vendor_id' => $topVendor->vendor_id,
                'vendor_name' => $topVendor->vendor->owner_name   ?? 'N/A',
                'completed_rfq_count' => $topVendor->completed_rfq_count,
            ] : null,

            'topDentistSeller' => $topdentist ? [
                'dentist_id' => $topdentist->user_id,
                'dentist_name' => $topdentist->user->owner_name   ?? 'N/A',
                'completed_rfq_count' => $topdentist->completed_rfq_count,
            ] : null,
        ]);
    }

    public function BankAccount(BankAccountRequest $request)
    {
        $user = Auth::user();

        $bankAccount = BankAccount::updateOrCreate(
            ['admin_id' => $user->id], // Match on user_id
            [
                'bank_account' => $request->input('bank_account'),
                'bank_holder_name' => $request->input('bank_holder_name'),
                'bank_number' => $request->input('bank_number'),
                'swift_code' => $request->input('swift_code'),
                'iban' => $request->input('iban'),
            ]
        );

        return $this->ApiResponse('200', __('success'),  __('Data Updated successfully.'));

    }
    public function getBankData()
    {
        $bankdata = BankAccount::where('admin_id' , auth()->user()->id)->select(['id','bank_account' , 'bank_holder_name' , 'bank_number' , 'swift_code' , 'iban'])->first();
        return $this->ApiResponse('200', __('success'),  __('success') , $bankdata);

    }
}
