<?php

namespace App\Models;

use App\Traits\GetAttribute;
use App\Traits\SearchableRelations;
use Illuminate\Database\Eloquent\Model;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\MediaLibrary\HasMedia;

class Product extends Model
{
    use HasFactory , GetAttribute , SearchableRelations;
    protected $fillable = ['name', 'description', 'price', 'user_id', 'main_category_id', 'sub_category_id'];
    protected $appends = ['is_in_cart']; // Automatically include this attribute in JSON responses

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);
        $this->multiple_attachment = true;
        $this->multiple_attachment_usage = ['default', 'bdf-file'];
    }


    public function getImageAttribute()
    {
        return $this->attachmentRelation()->first();
    }
    public function mainCategory(): BelongsTo
    {
        return $this->belongsTo(Category::class, 'main_category_id');
    }


    public function subCategory(): BelongsTo
    {
        return $this->belongsTo(Category::class, 'sub_category_id');
    }


    public function favoritedByUsers()
    {
        return $this->belongsToMany(User::class, 'favourites')->withTimestamps();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function rfqs()
    {
        return $this->hasMany(RFQ::class);
    }

    public function specs(): HasMany
    {
        return $this->hasMany(Spec::class);
    }

    public function getIsFavAttribute()
    {
        return $this->favoritedByUsers()->where('user_id', auth()->id())->exists();
    }


    public function getIsInCartAttribute()
    {
        $userId = auth()->id();

        if (!$userId) {
            return false; // No authenticated user
        }

        // Check if the product exists in the user's cart
        return \App\Models\CartItem::whereHas('cart', function ($query) use ($userId) {
            $query->where('user_id', $userId);
        })->where('product_id', $this->id)->exists();
    }

}
