<?php

namespace App\Traits;

use App\Traits\Logger;
use App\Traits\Searchful;
use App\helpers\Attachment;
use App\Traits\ErrorMessage;
use App\Models\Permission\Group;
use App\Traits\PaginatedResponse;
use Illuminate\Support\Facades\DB;
use Symfony\Component\HttpFoundation\Response;

// public trait for controllers to do all crud operations. you can customize it by overriding the method you want to customize.
trait HasCrudActions
{
    use ApiResponse, Logger, HistoryLogger, Searchful, PaginatedResponse, ErrorMessage;
    // get untrashed models
    public function index()
    {
        $model = app($this->model);

        // Use the method to retrieve relations for eager loading

        $relations = method_exists($model, 'getWithRelations') ? $model->getWithRelations() : [];

        $modelResource = $this->search()->with($relations);
        if (isset($this->hasPagination) && $this->hasPagination == false) {
            return $this->apiResponse('200', __('success'), '', $modelResource->orderBy('id', 'DESC')->get());
        }

        return $this->paginatedResponse($modelResource->orderBy('id', 'DESC'));
    }

    // get trashed models
    public function archive()
    {
        $recordsQuery = $this->search()->where('is_archived' , 0)->onlyTrashed();

        return $this->paginatedResponse($recordsQuery);
    }

    public function store()
    {
        try {
            $request = app($this->request);
            
            DB::beginTransaction();

            // Prepare data to create the record
            $recordData = $request->except('image');



            // Create the record
            $record = $this->model::create($recordData);

            // Attach media if provided
            if ($request->hasFile('image')) {
                Attachment::addAttachment($request->file('image'), $record, 'images', ['save' => 'original', 'usage' => 'image']);
            }



            // Create history

            DB::commit();

            // Return response
            $recordResource = new $this->resource($record);
            return $this->apiResponse(Response::HTTP_CREATED, __('success'), __('created successfully'), $recordResource);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->log(0, $this->model, $e->getMessage(),$e);
            return $this->errorMessage($e);
        }
    }


    public function show(string $id)
    {
        $record = $this->model::find($id);

        if (!$record) {
            return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
        }
        $relations = method_exists($record, 'getWithRelations') ? $record->getWithRelations() : [];

        $recordResource = new $this->resource($record->load($relations));

        return $this->apiResponse(Response::HTTP_OK, __('success'), '', $recordResource);
    }

    public function update(string $id)
    {
        try {
            $request = app($this->request);

            $record = $this->model::find($id);

            if (!$record) {
                return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
            }

            DB::beginTransaction();
            $record->update($request->except('image'));

             // Attach media if provided
             if ($request->hasFile('image')) {
                Attachment::updateAttachment($request->file('image') ,$record->image ,$record, 'images', ['save' => 'original', 'usage' => 'image']);
            }


            DB::commit();

            $recordResource = new $this->resource($record);

            return $this->apiResponse(Response::HTTP_OK, __('success'), __('Updated Successfully!'), $recordResource);
        } catch (\Exception $e) {

            DB::rollBack();

            $this->log(0, $this->model, $e->getMessage(),$e);

            return $this->errorMessage($e);
        }
    }

    // soft delete
    public function destroy(string $id )
    {
        try {
            $record = $this->model::find($id);
            if (!$record) {
                return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource_not_found'));
            }

            DB::beginTransaction();

            $record->delete();

            DB::commit();

            return $this->apiResponse(Response::HTTP_OK, __('success'), __('deleted successfully'));
        } catch (\Exception $e) {

            DB::rollBack();

            $this->log($id, $this->model, $e->getMessage(),$e);

            return $this->errorMessage($e);
        }
    }

    // delete for ever
    public function forceDelete(string $id)
    {
        try {
            $record = $this->model::onlyTrashed()->where('id', $id)->where('is_archived', 0)->first();
            if (!$record) {
                return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
            }

            DB::beginTransaction();



            $record->update([
                'is_archived' => true,
                'name' => "geex_deleted_" . fake()->randomNumber(6, true) . "_$record->name",
                'username' => "geex_deleted_" . fake()->randomNumber(6, true) . "_$record->username",


                'email' => "geex_deleted_" . fake()->randomNumber(6, true) . "_$record->email"
            ]);

            $this->createHistory($this->model, $record->id, 'deleted');

            DB::commit();

            return $this->apiResponse(Response::HTTP_OK, __('success'), __('deleted completely successfully'));
        } catch (\Exception $e) {

            DB::rollBack();

            $this->log($id, $this->model, $e->getMessage(),$e);

            return $this->errorMessage($e);
        }
    }

    // restore deleted model
    public function restore(string $id)
    {
        $record = $this->model::onlyTrashed()->where('id', $id)->where('is_archived', 0)->first();

        if (!$record) {
            return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
        }

        try {
            $record->restore();

            return $this->apiResponse(Response::HTTP_OK, __('success'), __('restored successfully'));
        } catch (\Exception $e) {

            DB::rollBack();

            $this->log($id, $this->model, $e->getMessage(),$e);

            return $this->errorMessage($e);
        }
    }
}
