<?php

namespace App\Http\Controllers\Api\Admin;

use App\Models\RFQ;
use App\Models\User;
use App\Services\FcmService;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Admin\OrderRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Resources\Api\Admin\OrderResource;

class OrderController extends Controller
{
    use HasCrudActions;
    protected $model = RFQ::class;

    protected $resource = OrderResource::class;
    protected $request = OrderRequest::class;


    public function index(Request $request)
    {
        $model = app($this->model);

        // Use the method to retrieve relations for eager loading
        $searchableFields = [
            'user' => ['owner_name' => $request->user],
            'vendor' => ['owner_name' => $request->vendor],
            'order_type' => ['order_type' => $request->type]
        ];

        $modelResource = $this->search()
            ->when($request->has('date'), function($q) use ($request) {
                $q->whereDate('created_at', $request->date);
            })
            ->where(function($q) {
                // If the order_type is 'rfq_order', exclude the ones with status 'pending'
                $q->where('order_type', '<>', 'rfq_order')
                  ->orWhere(function($q) {
                      $q->where('order_type', 'rfq_order')
                        ->where('vendor_id', '!=', null);
                  });
            })
            ->searchRelated($searchableFields)
            ->with([
                'product:id,name',
                'specs:id,key,value',
                'branch:id,name,location,city_id',
                'vendor:id,owner_name',
                'user:id,owner_name',


            ]);

        return $this->paginatedResponse($modelResource->orderBy('id', 'DESC'), $this->resource);
    }

    public function update(string $id, FcmService $fcmService)
    {
        try {
            $request = app($this->request);
    
            $record = $this->model::find($id);
    
            if (!$record) {
                return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
            }
    
            DB::beginTransaction();
    
            // Check if the status has changed
            $oldStatus = $record->status;
            $newStatus = $request->get('status');
            
            // Update the record except for the image
            $record->update($request->except('image'));
    
            DB::commit();
    
            // Send notification if status has changed
            if ($newStatus && $oldStatus !== $newStatus) {
                $user = $record->user; // Assuming the model has a `user` relation
                if ($user && $user->fcmTokens()->exists()) {
                    $fcmService->send(
                        $user->fcmTokens[0]->device_token,
                        __('Order Status Updated'),
                        __('Your order status has been changed to :status', ['status' => $newStatus]),
                        [
                            'record_id' => (string) $record->id,
                            'status' => $newStatus,
                            'action' => 'order_status_update'
                        ],
                        $user
                    );
                }
            }
    
            $recordResource = new $this->resource($record);
    
            return $this->apiResponse(Response::HTTP_OK, __('success'), __('Updated Successfully!'), $recordResource);
        } catch (\Exception $e) {
            DB::rollBack();
    
            $this->log(0, $this->model, $e->getMessage(), $e);
    
            return $this->errorMessage($e);
        }
    }

    public function show(string $id)
    {
        $record = $this->model::find($id);

        if (!$record) {
            return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
        }
        $relations = method_exists($record, 'getWithRelations') ? $record->getWithRelations() : [];

        $recordResource = new $this->resource($record->load([
            'product:id,name',
            'specs:id,key,value',
            'branch:id,name,location,city_id',
            'user:id,owner_name',
            'vendor:id,owner_name',

        ]));

        return $this->apiResponse(Response::HTTP_OK, __('success'), '', $recordResource);
    }

    public function assignOrder(Request $request, FcmService $fcmService)
    {
        $request->validate([
            'order_id' => 'required|exists:r_f_q_s,id',
            'vendor_id' => 'required|exists:users,id',
        ]);
    
        try {
            // Find the order and assign the vendor
            $order = RFQ::find($request->order_id);
            $order->vendor_id = $request->vendor_id;
            $order->save();
    
            // Find the vendor
            $vendor = User::find($request->vendor_id);
    
            // Send notification to the vendor
            if ($vendor && $vendor->fcmTokens()->exists()) {
                $fcmService->send(
                    $vendor->fcmTokens[0]->device_token,
                    __('New Order Assigned'),
                    __('An order has been assigned to you by the admin. Order ID: :order_id', ['order_id' => $order->id]),
                    [
                        'order_id' => (string) $order->id,
                        'action' => 'order_assigned'
                    ],
                    $vendor
                );
            }
    
            return $this->apiResponse(Response::HTTP_OK, __('success'), __('Order assigned successfully'));
        } catch (\Exception $e) {
            $this->log(0, RFQ::class, $e->getMessage(), $e);
            return $this->errorMessage($e);
        }
    }
    


}
