<?php

namespace App\Http\Controllers\Api;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Admin;
use App\Traits\Logger;
use App\Models\FcmToken;
use App\Mail\PasswordReset;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use App\Events\UserRegistered;
use App\Traits\HasCrudActions;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Http\Requests\Auth\LoginRequest;
use App\Http\Resources\Api\LoginResource;
use App\Http\Requests\Auth\RegisterRequest;
use App\Http\Requests\Auth\EmailVerifyRequest;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Requests\Auth\ResetPasswordRequest;
use App\Http\Requests\Auth\CheckResetCodeRequest;
use App\Http\Requests\Auth\ForgotPasswordRequest;

class AuthController extends Controller
{

    use ApiResponse, Logger;
    protected $user;

    public function __construct()
    {
        $this->user   = new User();
    }

    public function Register(RegisterRequest $request)
    {
        try {
            DB::beginTransaction();

            $user = $this->user->createNewUser($request, 'user');

            // Dispatch the UserRegistered event
            event(new UserRegistered($user));

            DB::commit();
            return $this->apiResponse(Response::HTTP_OK, 'success', __('Registration Successful!'));
        } catch (Exception $e) {
            DB::rollBack();

            $this->log(null, User::class, $e->getMessage(), $e);
            // Return a generic error message to the API response
            return $this->apiResponse(Response::HTTP_INTERNAL_SERVER_ERROR, 'failed', __('Server error!'));
        }
    }



    public function login(LoginRequest $request)
    {
        if ($request->type == 'admin') {
            $record = Admin::where('email', $request->email)->first();
        } else {
            $record = User::where('email', $request->email)->with('country')->first();
        }

        if (!$record || !Hash::check($request->password, $record->password)) {
            return $this->apiResponse(400, 'fail', 'invalid credentials');
        }
        if ($record->email_verified_at == null) {
            return $this->apiResponse('403', __('fail'), __('Access denied: User is not verified'), ['email_verified_at' => $record->email_verified_at]);
        }
        $token = $record->createToken('token')->plainTextToken;
        if(request('device_token')) {
            FcmToken::firstOrCreate([
                'user_type' => $record::class,
                'user_id' => $record->id,
                'device_token' => $request->device_token
            ]);       
         }
        $latest_user_token = DB::table('personal_access_tokens')->where('tokenable_id', $record->id)->latest()->first();

        DB::table('personal_access_tokens')->where('id', $latest_user_token->id)->update([
            'device_token' => $request->device_token,
            'app_locale' => $request->app_locale
        ]);

        return $this->apiResponse(200, 'success', 'logged in successfully', [
            'token' => $token,
            'user' => new LoginResource($record)
        ]);
    }

    public function verify(EmailVerifyRequest $request){

        $user = User::where('email', $request->email)->first();

       $resetCode = DB::table('password_reset_codes')->where('email', $request->email)->where('code', $request->code)->first();

        if (!$resetCode) {
            return $this->apiResponse(404, 'fail', __('invalid reset code'));
        }

        $latestCode = DB::table('password_reset_codes')->where('email', $request->email)->latest()->first();

        if ($latestCode->code == $user->verification_code) {
            return $this->apiResponse(404, 'fail', __('invalid reset code'));
        }


        if ($latestCode->code != $resetCode->code) {
            return $this->apiResponse(404, 'fail', __('invalid reset code'));
        }

        $codeDate = Carbon::parse($resetCode->created_at);
        $currentDate = Carbon::now();
        if ($codeDate->diffInMinutes($currentDate) > 2) {
            return $this->apiResponse(400, 'fail', __('expired reset code'));
        }

        $user->email_verified_at = now();
        $user->verification_code = $request->code;

        $user->save();

        return response()->json([
            'message' => __('Your account has been verified successfully')
        ]);
    }

    public function sendPasswordResetEmail(ForgotPasswordRequest $request)
    {
        if ($request->type == 'admin') {
            $record = Admin::where('email', $request->email)->first();
        } else {
            $record = User::where('email', $request->email)->first();
        }

        if (!$record) {
            return $this->apiResponse(404, 'fail', __('email not found'));
        }

        // Check if a reset code was generated within the last minute
        $lastResetCode = DB::table('password_reset_codes')
                            ->where('email', $request->email)
                            ->where('created_at', '>=', now()->subMinutes(2))
                            ->exists();

        if ($lastResetCode) {
            // Another reset code was generated less than a minute ago
            return $this->apiResponse(429, 'fail', __('Please wait for some time before requesting another reset code.'));
        }

        $recoveryCode = fake()->randomNumber(4, true);

        DB::table('password_reset_codes')->insert([
            'type' => $record::class,
            'email' => $request->email,
            'code' => $recoveryCode,
            'created_at' => now()
        ]);

        try {
            Mail::to($request->email)->send(new PasswordReset($recoveryCode));
        } catch (\Exception $e) {
            // Handle the mail error
            return $this->apiResponse(500, 'fail', __('mail didn\'t send'), ['error' => $e->getMessage()]);
        }

        return $this->apiResponse(200, 'success', __('reset password code sent'), ['code' => $recoveryCode, 'email' => $request->email]);
    }


    public function checkResetCode(CheckResetCodeRequest $request)
    {
        $resetCode = DB::table('password_reset_codes')->where('email', $request->email)->where('code', $request->code)->first();

        if (!$resetCode) {
            return $this->apiResponse(404, 'fail', __('invalid reset code'));
        }

        $latestCode = DB::table('password_reset_codes')->where('email', $request->email)->latest()->first();


        if ($latestCode->code != $resetCode->code) {
            return $this->apiResponse(404, 'fail', __('invalid reset code'));
        }

        $codeDate = Carbon::parse($resetCode->created_at);
        $currentDate = Carbon::now();
        if ($codeDate->diffInMinutes($currentDate) > 2) {
            return $this->apiResponse(400, 'fail', __('expired reset code'));
        }


        return $this->apiResponse(200, 'success', __('valid reset code') , ['email' =>$request->email]);
    }

    public function resetPassword(ResetPasswordRequest $request)
    {
        $resetCode = DB::table('password_reset_codes')->where('email', $request->email)->where('code', $request->code)->first();

        if (!$resetCode) {
            return $this->apiResponse(404, 'fail', __('invalid reset code'));
        }

        $latestCode = DB::table('password_reset_codes')->where('email', $request->email)->latest()->first();

        if ($latestCode->code != $resetCode->code) {
            return $this->apiResponse(404, 'fail', __('invalid reset code'));
        }

        $codeDate = Carbon::parse($resetCode->created_at);
        $currentDate = Carbon::now();

        // Check if the code is older than 2 minutes
        if ($currentDate->diffInMinutes($codeDate) > 2) {
            return $this->apiResponse(400, 'fail', __('expired reset code'));
        }

        $user = $resetCode->type::where('email', $resetCode->email)->first();

        $user->update([
            'password' => bcrypt($request->password)
        ]);

        return $this->apiResponse(200, 'success', __('Password changed successfully.'));
    }



    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return $this->apiResponse(200, 'success', __('logged out successfully'));
    }
}
