<?php

namespace App\Http\Resources\Api\Admin;

use App\helpers\Attachment;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Http\Resources\MediaResource;
use App\Http\Resources\OrderItemResource;
use Illuminate\Http\Resources\Json\JsonResource;

class OrderResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'qty' => $this->qty,
            'status' => $this->status,
            'price' => Attachment::convertPrice($this->price),
            'type' => $this->order_type,
            'created_at' => Carbon::parse($this->created_at)->format('y-m-d h:i:s'),
            'bank_image' => MediaResource::make($this->image),
            'payment_method' => $this->payment_method,

            // Conditionally set the 'product' for 'rfq_order'
            'product' => $this->when(
                $this->order_type == 'rfq_order',
                $this->whenLoaded('product', function () {
                    return [
                        'id' => $this->product->id,
                        'name' => $this->product->name,
                        'image' => MediaResource::make($this->product->image), // Accessor for image
                    ];
                })
            ),

            // Conditionally set 'specs' for 'rfq_order'
            'specs' => $this->when(
                $request->routeIs('orders.show') && $this->order_type == 'rfq_order',
                $this->whenLoaded('specs', function () {
                    return $this->specs->map(function ($spec) {
                        return [
                            'id' => $spec->id,
                            'key' => $spec->key,
                            'value' => $spec->value,
                        ];
                    });
                })
            ),

            // Conditionally set the 'product' for 'direct_order'
            'products' => $this->when(
                $request->routeIs('orders.show') && $this->order_type == 'direct_order',
                OrderItemResource::collection($this->rfqItems)
            ),

            // Branch details
            'branch' => $this->when(
                $request->routeIs('orders.show'),
                $this->whenLoaded('branch', function () {
                    return [
                        'id' => $this->branch->id,
                        'location' => $this->branch->location,
                    ];
                })
            ),

            // User details
            'user' => $this->whenLoaded('user', function () {
                return [
                    'id' => $this->user->id,
                    'owner_name' => $this->user->owner_name,
                ];
            }),

            // Vendor details
            'vendor' => $this->whenLoaded('vendor', function () {
                return [
                    'id' => $this->vendor->id,
                    'owner_name' => $this->vendor->owner_name,
                ];
            }),
        ];
    }

}
