<?php

namespace App\Services;

// install google auth library
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use App\Notifications\FcmNotification;
use Google\Auth\Credentials\ServiceAccountCredentials;

class FcmService
{
    private string $auth_token;
    private string $project_id;

    public function __construct()
    {
        $this->auth_token = $this->getAuthToken();
        $this->project_id = json_decode(file_get_contents(storage_path('firebase-service-account.json')))->project_id;
    }

    private function getAuthToken()
    {
        $credentialsPath = storage_path('firebase-service-account.json');

        // Scopes required for FCM (Firebase Cloud Messaging)
        $scopes = ['https://www.googleapis.com/auth/firebase.messaging'];

        // Initialize Service Account Credentials using the JSON file
        $serviceAccount = new ServiceAccountCredentials($scopes, $credentialsPath);

        return $serviceAccount->fetchAuthToken()['access_token'];
    }

    public function send(string $token, string $title, string $body = null, array $data = [], $user)
    {
        $params = [
            'message' => [
                'token' => $token,
                'notification' => [
                    'title' => $title,
                    'body' => $body
                ],
                'android' => [
                    'direct_boot_ok' => true
                ]
            ]
        ];

        if (!empty($data) && !array_is_list($data)) {
            $params['message']['data'] = $data;
        }
        $user->notify(new FcmNotification($title, $body, $data));

        // Send the FCM notification
        $response = Http::withHeaders([
            'Authorization' => "Bearer {$this->auth_token}",
            'Content-Type' => 'application/json'
        ])->post("https://fcm.googleapis.com/v1/projects/{$this->project_id}/messages:send", $params);

        // Handle success or failure and log accordingly
        if ($response->successful()) {
            // Log success
            Log::info("FCM Notification sent successfully to user ID: {$user->id}", [
                'title' => $title,
                'body' => $body,
                'user_id' => $user->id,
                'data' => $data
            ]);
        } else {
            // Log failure
            Log::error("Failed to send FCM notification to user ID: {$user->id}", [
                'error' => $response->json(),
                'title' => $title,
                'body' => $body,
                'user_id' => $user->id,
                'data' => $data
            ]);
        }

        return $response;
    }
}
