<?php

namespace App\Http\Controllers\Api\Admin;

use App\Models\User;
use App\helpers\Attachment;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Admin\UserRequest;
use App\Http\Resources\Api\Admin\UserResource;
use Symfony\Component\HttpFoundation\Response;

class UserController extends Controller
{
    use HasCrudActions;


    protected $model = User::class;

    protected $resource = UserResource::class;

    protected $request = UserRequest::class;


    public function index(Request $request)
    {

        $searchableFields = [
            'type' => ['type' => $request->type]
        ];
        $modelResource = $this->search()->with('country:id,abbreviation,code');
        return $this->paginatedResponse($modelResource->searchRelated($searchableFields)->orderBy('id' , 'DESC') , $this->resource);

    }

    public function store()
    {
        try {
            $request = app($this->request);

            DB::beginTransaction();

            $recordData = $request->except('image' , 'type');
            $record = $this->model::create($recordData);
            $record->type = $request->type;
            $record->email_verified_at = now();
            $record->save();

            if ($request->hasFile('image')) {
                Attachment::addAttachment($request->file('image'), $record, 'users/profile', ['save' => 'original', 'usage' => 'image']);
            }


            DB::commit();

            // Return response
            $recordResource = new $this->resource($record);
            return $this->apiResponse(Response::HTTP_CREATED, __('success'), __('created successfully'), $recordResource);
        } catch (\Exception $e) {
            DB::rollBack();
            $this->log(0, $this->model, $e->getMessage(),$e);
            return $this->errorMessage($e);
        }
    }

    public function update(string $id)
    {
        try {
            $request = app($this->request);

            $record = $this->model::find($id);

            if (!$record) {
                return $this->apiResponse(Response::HTTP_OK, 'fail', __('resource not found'));
            }

            DB::beginTransaction();
            $record->update($request->except('image'));

             // Attach media if provided
             if ($request->hasFile('image')) {
                Attachment::updateAttachment($request->file('image'), $record,$record->image , 'images', ['save' => 'original', 'usage' => 'image']);
            }

            if ($request->has('bank_account')) {
                $bankAccountData = [
                    'bank_account' => $request->bank_account,
                    'bank_holder_name' => $request->bank_holder_name,
                    'bank_number' => $request->bank_number,
                    'swift_code' => $request->swift_code,
                    'iban' => $request->iban,
                ];

                $record->bankAccount()->updateOrCreate(
                    ['id' => $record->bankAccount ? $record->bankAccount->id : null], // Safely access id
                    $bankAccountData
                );
            }





            DB::commit();

            $recordResource = new $this->resource($record);

            return $this->apiResponse(Response::HTTP_OK, __('success'), __('Updated Successfully!'), $recordResource);
        } catch (\Exception $e) {

            DB::rollBack();

            $this->log(0, $this->model, $e->getMessage(),$e);

            return $this->errorMessage($e);
        }
    }


    public function productRequests()
    {
        $productAccess= User::where('product_access' , 'pending')->get(['id' , 'owner_name' , 'product_access']);

        return $this->apiResponse(Response::HTTP_OK, __('success'), __('Success'), $productAccess);

    }




}
