<?php
namespace App\Imports;

use App\Models\Spec;
use App\Models\Product;
use App\Models\Category;
use App\helpers\Attachment;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class ProductsImport implements ToModel, WithHeadingRow
{
    public function model(array $row)
    {
        // Skip the row if the 'name' is missing or empty
        if (empty($row['name'])) {
            return null;
        }

        // Find or create the main category
        $mainCategory = Category::where('name', 'LIKE', '%' . $row['main_category'] . '%')->first();
        if (!$mainCategory) {
            $mainCategory = Category::create([
                'name' => $row['main_category'],
                'description' => 'Imported category',
            ]);
        }

        // Create the product
        $product = Product::create([
            'name' => $row['name'],
            'description' => $row['description'] ?? null, // Use null if description is missing
            'price' => $row['price'] ?? 0.00, // Default to 0.00 if price is missing
            'main_category_id' => $mainCategory->id,
            'sub_category_id' => $mainCategory ? $mainCategory->id : null, // Use the same ID for subcategory for now
        ]);

        // Handle image upload (if provided)
        if (!empty($row['image'])) {
                // Convert the local file path to an UploadedFile object
                $uploadedFile = $this->createUploadedFile($row['image']);
                    // Associate the image with the product using your Attachment logic
                    Attachment::addAttachment($uploadedFile, $product, 'products/images', ['save' => 'original', 'usage' => 'image']);
             
          
        }

        // Create the specs (if provided)
        if (!empty($row['specs'])) {
            $specs = json_decode($row['specs'], true); // Assuming specs are stored as JSON in the Excel sheet
            foreach ($specs as $spec) {
                Spec::create([
                    'product_id' => $product->id,
                    'key' => $spec['key'],
                    'value' => $spec['value'],
                ]);
            }
        }

        return $product;
    }

    /**
     * Convert a local file path to an UploadedFile object.
     *
     * @param string $filePath
     * @return UploadedFile|null
     */
    private function createUploadedFile($filePath)
    {
        if (!File::exists($filePath)) {
            Log::error("File does not exist: " . $filePath);
            return null;
        }

        $maxFileSize = 10 * 1024 * 1024; // 10 MB
        if (filesize($filePath) > $maxFileSize) {
            Log::error("File is too large: " . $filePath);
            return null;
        }
    
        // Check file type
        $allowedExtensions = ['png', 'jpg', 'jpeg'];
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        if (!in_array(strtolower($extension), $allowedExtensions)) {
            Log::error("Unsupported file type: " . $filePath);
            return null;
        }

        // Get the file name and extension
        $fileName = basename($filePath);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);

        // Create a temporary file path
        $tempPath = tempnam(sys_get_temp_dir(), 'imported_image');
        if (!copy($filePath, $tempPath)) {
            Log::error("Failed to copy file to temporary location: " . $filePath);
            return null;
        }

        // Create an UploadedFile object
        return new UploadedFile(
            $tempPath,
            $fileName,
            mime_content_type($tempPath),
            filesize($tempPath),
            false, // Do not move the file
            true   // Test mode
        );
    }
}