<?php

namespace App\Http\Controllers\Admin;

use App\Http\Resources\Admin\AdminNotificationReource;
use App\Models\User;
use App\Services\FcmService;
use Illuminate\Http\Request;
use App\Traits\HasCrudActions;
use App\Models\AdminNotification;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;

class NotificationController extends Controller
{
    use HasCrudActions;


    public function index()
    {
        return $this->paginatedResponse(AdminNotification::orderBy('id', 'DESC') , AdminNotificationReource::class);
    }


    public function sendNotification(Request $request, FcmService $fcmService)
    {
        $validatedData = $request->validate([
            'target_type' => 'required|in:user,vendor,dentist,all',
            'notification_type' => 'required|string',
            'body' => 'required|string',
            'receiver_ids' => 'nullable|array',
        ]);
    
        $users = [];
        switch ($validatedData['target_type']) {
            case 'user':
                $userId = $request->get('user_id'); // Specific user ID passed in request
                $users = User::where('id', $userId)->with('fcmTokens')->get();
                break;
    
            case 'vendor':
                $users = User::where('type', 'vendor')->with('fcmTokens')->get();
                break;
    
            case 'dentist':
                $users = User::where('type', 'dentist')->with('fcmTokens')->get();
                break;
    
            case 'all':
                $users = User::with('fcmTokens')->get();
                break;
        }
    
        // Filter users with FCM tokens
        $usersWithTokens = $users->filter(function ($user) {
            return $user->fcmTokens->isNotEmpty();
        });
    
        $adminNotification = AdminNotification::create([
            'sender_id' => auth()->id(),
            'title' => $validatedData['notification_type'],
            'body' => $validatedData['body'],
            'receivers_type' => $validatedData['target_type'],
        ]);
    
        foreach ($usersWithTokens as $user) {
            foreach ($user->fcmTokens as $token) {
                if ($token->device_token) {
                    $fcmService->send(
                        $token->device_token,
                        $validatedData['notification_type'],
                        $validatedData['body'],
                        ['user_id' => (string) $user->id, 'action' => 'notification'],
                        $user
                    );
                }
            }
        }
    
        return response()->json([
            'message' => 'Notification sent successfully',
        ], 200);
    }
    
    
}
