<?php

namespace App\Traits;

use App\Models\Log;
use Illuminate\Support\Facades\Log as FileLog;
use Illuminate\Support\Facades\Mail;


trait Logger
{
    public function log($model_id, $model_type, $message, $exception = null)
    {
        $user = auth()->user();
        $user_id = $user?->id;
        $user_type = $user ? $user::class : '';

        try {
            Log::create([
                'user_type' => $user_type,
                'user_id' => $user_id,
                'model_type' => $model_type,
                'model_id' => $model_id,
                'message' => $message
            ]);
        } catch (\Exception $e) {
            FileLog::channel('details_log')->error("Failed to create log entry in database: " . $e->getMessage());
        }

        $debug_backtrace = debug_backtrace();
        $callingFunction = $debug_backtrace[1]['function'];
        $file = $debug_backtrace[0]['file'];
        $line = $debug_backtrace[0]['line'];
        $class = $debug_backtrace[0]['class'];

        $logMessage = sprintf(
            "\nFile: %s\nClass: %s\nFunction: %s\nLine: %s\nMessage: %s\nRequest: %s\n",
            $file,
            $class,
            $callingFunction,
            $line,
            $message,
            json_encode(request()->all())
        );

        if ($exception) {
            $logMessage .= sprintf(
                "Exception Message: %s\nException File: %s\nException Line: %s\n",
                $exception->getMessage(),
                $exception->getFile(),
                $exception->getLine()
            );
        }

        if ($user) {
            $logMessage .= sprintf("User: %s\nUser Type: %s\nUser ID: %s\n", json_encode($user), $user_type, $user_id);
        }

        if ($model_type && $model_id) {
            $logMessage .= sprintf("Model Type: %s\n", $model_type);
        }

        if ($model_id) {
            $logMessage .= sprintf("Model ID: %s\n", $model_id);
        }

        FileLog::channel('details_log')->error($logMessage);

        $this->sendExceptionEmail($file, $callingFunction, $message, $class, $line);
    }




    public function sendExceptionEmail($filePath,$function,$message,$class)
    {
        $data = [
            'fileName' => $filePath,
            'function' => $function,
            'messages' => $message,
            'class' => $class,
        ];

        try {
            Mail::send('emails.exception', $data, function ($message) {
                $message->to('amro7275@gmail.com')
                        ->subject('Exception Occurred');
            });
        } catch (\Exception $e) {
            FileLog::channel('details_log')->error("Failed to send exception email: " . $e->getMessage());
        }
    }

}
